#include <stdint.h>
#include <stdio.h>
#include "dhcpserver/dhcpserver_options.h"
// Includes de FreeRTOS
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/semphr.h" // Necesario para el Mutex

#include "esp_pm.h"
#include "esp_err.h"
#include "fontx.h"
#include "driver/gpio.h"
#include "driver/spi_master.h"
#include "esp_log.h"
#include "hal/adc_types.h"
#include "hal/ledc_types.h"
#include "portmacro.h"
#include "st7789.h"
#include "driver/adc.h"
#include "driver/ledc.h"
#include "esp_spiffs.h"
#include "esp_adc/adc_continuous.h"
#include "driver/adc_types_legacy.h" 


// ======= PINES / DEFINES =======
#define PIN_NUM_MOSI 23
#define PIN_NUM_CLK  18
#define PIN_NUM_CS   -1
#define PIN_NUM_DC   19
#define PIN_NUM_RST  33
#define PIN_NUM_BL   4

// botones
#define PIN_ARMAR   22
#define PIN_APAGAR  21

// ESC
#define ESC_PIN 25
#define POTE_ESC ADC1_CHANNEL_0   // pot del ESC

// Servo 1 + pote del servo
#define SERVO_PIN 26 // Primer Servo
#define SERVO_PIN_2 27 // ¡NUEVO! Segundo Servo (cambia este pin si es necesario)
#define POTE_SERVO ADC1_CHANNEL_3 // por ejemplo GPIO39

// Display/fuente globals
TFT_t tft;
FontxFile fx[1];
char buffer[32];

// ======= ESTADO COMPARTIDO =======
// Estas variables son accedidas por ambas tareas y deben ser protegidas
bool motor_armado = false;
int adc_esc_raw = 0;
int adc_servo_raw = 0;

// Mutex para proteger el estado compartido
SemaphoreHandle_t state_mutex;

// Constantes PWM (calculadas para 15-bit @ 50Hz)
const int PWM_MAX_15BIT = (1 << 15) - 1; // 32767
const float PWM_PERIOD_MS = 20.0f; // Periodo de 50Hz
const float ESC_PULSE_MIN_MS = 1.0f;
const float ESC_PULSE_MAX_MS = 2.0f;
const float SERVO_PULSE_MIN_MS = 0.5f;
const float SERVO_PULSE_MAX_MS = 2.5f;

// ======= MODIFICACIONES PARA EL ARMADO DEL ESC =======

/**
 * @brief Convierte un pulso en ms a un valor de 'duty' de 15 bits.
 */
int ms_a_duty(float pulse_ms) {
    // Duty = (Ancho de pulso / Periodo PWM) * Max_Duty
    return (int)((pulse_ms / PWM_PERIOD_MS) * PWM_MAX_15BIT + 0.5f);
}

/**
 * @brief Envía el pulso PWM MÍNIMO (1ms) al ESC.
 * Esto se usa para detener el motor y para la fase de calibración/armado.
 */
void enviar_pwm_minimo() {
    int min_duty = ms_a_duty(ESC_PULSE_MIN_MS);
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0, min_duty);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0);
}

/**
 * @brief Envía el pulso PWM MÁXIMO (2ms) al ESC.
 * Esto se usa SÓLO en la fase inicial de calibración del ESC.
 */
void enviar_pwm_maximo() {
    int max_duty = ms_a_duty(ESC_PULSE_MAX_MS);
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0, max_duty);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0);
}

/**
 * @brief Implementa el protocolo de armado inicial: MAXIMO -> MINIMO.
 * Es crucial para que el ESC entienda el rango de la señal.
 * ¡Debe ejecutarse antes de crear las tareas de FreeRTOS!
 */
void calibrar_esc() {
    ESP_LOGI("ESC", "Iniciando secuencia de calibración (MAXIMO -> MINIMO).");
    
    // 1. Enviar pulso MÁXIMO (2ms) - El ESC pitará el tono de máximo.
    enviar_pwm_maximo();
    // Esperar un tiempo suficiente para que el ESC detecte el máximo.
    vTaskDelay(pdMS_TO_TICKS(3000)); // Esperar 3 segundos (configurable)

    // 2. Enviar pulso MÍNIMO (1ms) - El ESC debería confirmar el armado con un pitido final.
    enviar_pwm_minimo();
    // Esperar a que el ESC complete la secuencia de armado.
    vTaskDelay(pdMS_TO_TICKS(3000)); // Esperar 3 segundos (configurable)
    
    ESP_LOGI("ESC", "Secuencia de calibración/armado completada. El motor no debería pitar más.");
}


// La función 'enviar_pwm_armar' ahora es redundante y se elimina, 
// ya que 'enviar_pwm_minimo' es lo que mantiene el motor a cero RPM.


// ======= FUNCIONES DE INICIALIZACIÓN =======
void montar_spiffs() {
    esp_vfs_spiffs_conf_t conf = {
        .base_path = "/spiffs",
        .partition_label = "spiffs",
        .max_files = 5,
        .format_if_mount_failed = true
    };
    esp_err_t ret = esp_vfs_spiffs_register(&conf);
    if (ret != ESP_OK) {
        ESP_LOGE("SPIFFS", "Error montando SPIFFS: %s", esp_err_to_name(ret));
    } else {
        ESP_LOGI("SPIFFS", "SPIFFS montado en /spiffs");
    }
}

void init_display() {
    montar_spiffs();
    InitFontx(fx, "/spiffs/ILGH32XB.FNT", NULL);
    st7789spi_master_init(&tft, PIN_NUM_MOSI, PIN_NUM_CLK, PIN_NUM_CS, PIN_NUM_DC, PIN_NUM_RST, PIN_NUM_BL);
    lcdInit(&tft, 240, 240, 0, 0);
    lcdFillScreen(&tft, BLACK);

    // Backlight
    if (PIN_NUM_BL != -1) {
        gpio_set_direction(PIN_NUM_BL, GPIO_MODE_OUTPUT);
        gpio_set_level(PIN_NUM_BL, 1);
    }
}

void init_adc() {
    adc1_config_channel_atten(POTE_ESC, ADC_ATTEN_DB_12);
    adc1_config_channel_atten(POTE_SERVO, ADC_ATTEN_DB_12);
    adc1_config_width(ADC_BITWIDTH_12);
}

void init_buttons() {
    // Configuración para usar PULLUP (botón a GND al presionar)
    gpio_set_direction(PIN_ARMAR, GPIO_MODE_INPUT);
    gpio_set_pull_mode(PIN_ARMAR, GPIO_PULLUP_ONLY);
    gpio_set_direction(PIN_APAGAR, GPIO_MODE_INPUT);
    gpio_set_pull_mode(PIN_APAGAR, GPIO_PULLUP_ONLY);
}

void init_pwm() {
    ledc_timer_config_t ledc_timer = {
        .speed_mode       = LEDC_HIGH_SPEED_MODE,
        .timer_num        = LEDC_TIMER_0,
        .duty_resolution  = LEDC_TIMER_15_BIT,
        .freq_hz          = 50,
        .clk_cfg          = LEDC_AUTO_CLK
    };
    ledc_timer_config(&ledc_timer);

    // Canal 0 para el ESC
    ledc_channel_config_t esc_channel = {
        .gpio_num   = ESC_PIN, .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel    = LEDC_CHANNEL_0, .timer_sel  = LEDC_TIMER_0,
        .duty       = 0, .hpoint     = 0
    };
    ledc_channel_config(&esc_channel);

    // Canal 1 para el Servo 1
    ledc_channel_config_t servo_channel_1 = {
        .gpio_num   = SERVO_PIN, .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel    = LEDC_CHANNEL_1, .timer_sel  = LEDC_TIMER_0,
        .duty       = 0, .hpoint     = 0
    };
    ledc_channel_config(&servo_channel_1);
    
    // ¡NUEVO! Canal 2 para el Servo 2
    ledc_channel_config_t servo_channel_2 = {
        .gpio_num   = SERVO_PIN_2, .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel    = LEDC_CHANNEL_2, .timer_sel  = LEDC_TIMER_0,
        .duty       = 0, .hpoint     = 0
    };
    ledc_channel_config(&servo_channel_2);
}

// ======= FUNCIONES DE CONTROL (Llamadas desde Tarea 1) =======

// Procesa botones y actualiza el estado (protegido por mutex)
void procesar_botones() {
    // Los pines están configurados con PULLUP, por lo que 0 significa presionado.
    int btn_armar = gpio_get_level(PIN_ARMAR);
    int btn_apagar = gpio_get_level(PIN_APAGAR);

    // Tomamos el control del mutex para modificar el estado
    if (xSemaphoreTake(state_mutex, pdMS_TO_TICKS(10)) == pdTRUE) {
        
        // Botón ARMAR presionado (0) y motor no armado
        if (btn_armar == 0 && !motor_armado) {
            ESP_LOGI("CONTROL", "Motor ARMADO (listo para acelerar)");
            motor_armado = true;
            // No se necesita enviar PWM aquí, la tarea de control lo hará.
        }

        // Botón APAGAR presionado (0) y motor armado
        if (btn_apagar == 0 && motor_armado) {
            ESP_LOGI("CONTROL", "Motor APAGADO (cero RPM)");
            motor_armado = false;
            enviar_pwm_minimo(); // Detiene el motor inmediatamente
        }
        
        // Liberamos el mutex
        xSemaphoreGive(state_mutex);
    }
}

// Procesa ADC y actualiza el estado (protegido por mutex)
void procesar_adc() {
    // Tomamos el control del mutex para modificar el estado
    if (xSemaphoreTake(state_mutex, pdMS_TO_TICKS(10)) == pdTRUE) {
        // La lectura de ADC puede tomar tiempo, por eso es bueno hacerlo fuera del mutex
        // Sin embargo, para este ejemplo simple, se deja así.
        adc_esc_raw = adc1_get_raw(POTE_ESC);
        adc_servo_raw = adc1_get_raw(POTE_SERVO);
        // Liberamos el mutex
        xSemaphoreGive(state_mutex);
    }
}

// Actualiza el PWM del ESC (basado en el estado)
void actualizar_esc_pwm() {
    int local_adc_esc;
    bool local_motor_armado;

    // Tomamos el mutex para LEER el estado
    if (xSemaphoreTake(state_mutex, pdMS_TO_TICKS(10)) == pdTRUE) {
        local_adc_esc = adc_esc_raw;
        local_motor_armado = motor_armado;
        xSemaphoreGive(state_mutex);
    } else {
        return; // No pudimos leer, saltamos este ciclo
    }

    // Si el motor no está armado, aseguramos que esté en la señal mínima (debería estarlo)
    if (!local_motor_armado) {
        // La función procesar_botones ya llama a enviar_pwm_minimo() al apagar.
        // Si no está armado por defecto, se mantendrá el mínimo.
        return;
    }

    // El motor está armado, ahora leemos el potenciómetro y ajustamos la velocidad
    int min_duty = ms_a_duty(ESC_PULSE_MIN_MS);
    int max_duty = ms_a_duty(ESC_PULSE_MAX_MS);
    
    // Mapeamos el ADC (0-4095) al rango de Duty (min_duty a max_duty)
    int duty = min_duty + (local_adc_esc * (max_duty - min_duty)) / 4095;
    
    // Seguridad: Si el pote está en 0, nos aseguramos de que sea el pulso mínimo
    if (local_adc_esc == 0) {
        duty = min_duty;
    }

    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0, duty);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_0);
}

// Actualiza el PWM de los dos Servos (basado en el estado)
void actualizar_servo_pwm() {
    int local_adc_servo;

    // Tomamos el mutex para LEER el estado
    if (xSemaphoreTake(state_mutex, pdMS_TO_TICKS(10)) == pdTRUE) {
        local_adc_servo = adc_servo_raw;
        xSemaphoreGive(state_mutex);
    } else {
        return; // No pudimos leer, saltamos este ciclo
    }

    // Calculamos los valores de duty para el rango del servo
    int duty_min = ms_a_duty(SERVO_PULSE_MIN_MS);
    int duty_max = ms_a_duty(SERVO_PULSE_MAX_MS);
    int duty_range = duty_max - duty_min;
    
    // Cálculo del deber para el Servo 1 (original)
    int duty_servo_1 = duty_min + (local_adc_servo * duty_range) / 4095;
    
    // Cálculo del deber para el Servo 2 (inverso)
    int local_adc_servo_inverso = 4095 - local_adc_servo;
    int duty_servo_2 = duty_min + (local_adc_servo_inverso * duty_range) / 4095;


    // Aplicar PWM al Servo 1 (Canal 1)
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_1, duty_servo_1);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_1);
    
    // Aplicar PWM al Servo 2 (Canal 2)
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2, duty_servo_2);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2);
}

// ======= FUNCIONES DE UI (Llamadas desde Tarea 2) =======

// Dibuja el estado (ARMADO/APAGADO)
void dibujar_estado(bool armado) {
    // Borra el área completa donde se dibuja el texto de estado
    // (Asegúrate de que este borrado no toque la línea de Y=239)
    lcdDrawFillRect(&tft, 70, 240, 240, 260, BLACK); 
    
    if (armado) {
        // La posición Y=242 está justo debajo de la línea Y=239
        lcdDrawString(&tft, fx, 80, 242, (uint8_t *)"ARMADO", GREEN); 
    } else {
        lcdDrawString(&tft, fx, 70, 242, (uint8_t *)"APAGADO", RED);
    }
}

// Dibuja el gráfico del ESC (solo actualiza el número)
void dibujar_esc_grafico(int adc_val) {
    static int prev_radius = 0; // Estático para recordar el radio anterior
    int radius = (adc_val * 60) / 4095 + 10;

    // Borra el círculo anterior
    if (prev_radius > 0) {
        lcdDrawCircle(&tft, 120, 90, prev_radius, BLACK);
    }
    
    // Dibuja el nuevo círculo 
    lcdDrawCircle(&tft, 120, 90, radius, GREEN);
    
    // Generar SÓLO el número. La palabra "POWER:" ya es estática.
    // Mapeamos 0-4095 a 0-99%
    int porcentaje = (adc_val * 100) / 4095;
    sprintf(buffer, "%02d%%", porcentaje); 
    
    // Limpia SOLO el espacio de los dos dígitos (Ej: posición X=130 a X=200, Y=200 a Y=240)
    // El texto "POWER:" se dibuja en Y=200. El número va a su lado.
    lcdDrawFillRect(&tft, 160, 166, 230, 209, BLACK); 

    // Dibujar SÓLO el número. Empieza después de la palabra estática "POWER:"
    lcdDrawString(&tft, fx, 160, 200, (uint8_t *)buffer, WHITE); 

    prev_radius = radius;
}

// ======= TAREAS DE FREERTOS =======

/**
 * @brief Tarea de alta prioridad para control de entradas y PWM.
 */
static void input_control_task(void *pvParameters) {
    ESP_LOGI("TASKS", "Iniciando input_control_task");
    while (true) {
        procesar_botones();
        procesar_adc();
        actualizar_esc_pwm();
        actualizar_servo_pwm(); // Ahora controla los dos servos

        // Se ejecuta 50 veces por segundo (cada 20ms)
        vTaskDelay(pdMS_TO_TICKS(20)); 
    }
}

/**
 * @brief Tarea de baja prioridad para actualizar la pantalla.
 */
static void display_task(void *pvParameters) {
    ESP_LOGI("TASKS", "Iniciando display_task");
    
    bool local_motor_armado = false; 
    int local_adc_esc = 0;           
    bool last_motor_state; 
    
    // Variables para saber si el estado cambió y evitar redibujar
    // Leemos el valor global una vez de forma segura para inicializar last_motor_state
    if (xSemaphoreTake(state_mutex, portMAX_DELAY) == pdTRUE) {
        last_motor_state = !motor_armado; 
        xSemaphoreGive(state_mutex);
    } else {
        last_motor_state = true; 
    }


    // **********************************************
    //          DIBUJO DE INTERFAZ ESTÁTICA
    // **********************************************
    lcdDrawString(&tft, fx, 0, 30, (uint8_t *)"R2D2", GREEN);
    
    // PALABRA POWER: ESTÁTICA
    lcdDrawString(&tft, fx, 56, 200, (uint8_t *)"POWER:", WHITE); 

    // 1. Línea entre el Círculo (área central) y la palabra POWER
    // Va de X=0 a X=240, en Y=165
    lcdDrawLine(&tft, 0, 165, 240, 165, WHITE); 

    // 2. Línea entre la palabra POWER y la palabra de ESTADO (ARMADO/APAGADO)
    // Va de X=0 a X=240, en Y=239
    lcdDrawLine(&tft, 0, 239, 240, 239, WHITE);

    while (true) {
        // 1. Obtener una copia segura del estado compartido (Mutex)
        if (xSemaphoreTake(state_mutex, portMAX_DELAY) == pdTRUE) {
            local_motor_armado = motor_armado;
            local_adc_esc = adc_esc_raw;
            xSemaphoreGive(state_mutex);
        }

        // 2. Actualizar el estado (solo si cambió)
        if (local_motor_armado != last_motor_state) {
            dibujar_estado(local_motor_armado);
            last_motor_state = local_motor_armado;
        }

        // 3. Actualizar el gráfico del ESC (círculo y número)
        dibujar_esc_grafico(local_adc_esc);

        // Se ejecuta 20 veces por segundo (cada 50ms)
        vTaskDelay(pdMS_TO_TICKS(50));
    }
}

// ====== app_main (Ahora solo configura e inicia tareas) ======
void app_main(void) {
    // 1. Inicializaciones
    init_display();
    init_adc();
    init_buttons();
    init_pwm();

    // 2. Crear el Mutex
    state_mutex = xSemaphoreCreateMutex();
    if (state_mutex == NULL) {
        ESP_LOGE("MAIN", "Error al crear el mutex. Abortando.");
        return; // No continuar si el mutex falla
    }
    
    // *************************************************************
    // 3. CALIBRACIÓN/ARMADO DEL ESC (antes de las tareas)
    //    Esto es lo que soluciona el problema de que el motor pite
    // *************************************************************
    calibrar_esc(); 

    // 4. Crear las tareas
    xTaskCreate(
        input_control_task,    // Función de la tarea
        "control_task",        // Nombre (para debug)
        4096,                  // Tamaño de la pila (stack)
        NULL,                  // Parámetros de la tarea
        10,                    // Prioridad (alta)
        NULL                   // Handle de la tarea (opcional)
    );

    xTaskCreate(
        display_task,          // Función de la tarea
        "display_task",        // Nombre (para debug)
        4096,                  // Tamaño de la pila (stack)
        NULL,                  // Parámetros de la tarea
        5,                     // Prioridad (baja)
        NULL                   // Handle de la tarea (opcional)
    );

    ESP_LOGI("MAIN", "Inicialización completada. Tareas iniciadas.");
}